<?php
namespace Mint\Service\Repositories;

ini_set('max_execution_time', 0);

use App\Models\User;
use App\Traits\Install;
use App\Helpers\IpHelper;
use App\Helpers\SysHelper;
use App\Models\Config\Config;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\ValidationException;
use Minishlink\WebPush\VAPID;

class InstallRepository
{
    use Install;

    /**
     * Instantiate a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
    }

    /**
     * Force migrate
     */
    public function forceMigrate() : string
    {
        if (SysHelper::getApp('INSTALLED')) {
            return 'Could not migrate!';
        }
        \Artisan::call('migrate', ['--force' => true]);
        return 'Migration completed!';
    }

    /**
     * Check all pre-requisite for script
     */
    public function getPreRequisite() : array
    {
        $pre_requisite = $this->installPreRequisite();
        $app = array(
            'verifier' => config('app.verifier'),
            'name'     => config('app.name'),
            'version'  => SysHelper::getApp('VERSION')
        );
        return compact('pre_requisite', 'app');
    }

    /**
     * Validate database connection, table count
     */
    public function validateDatabase() : bool
    {
        $link = @mysqli_connect(
            request('db_host'),
            request('db_username'),
            request('db_password'),
            request('db_database'),
            request('db_port')
        );

        if (! $link) {
            throw ValidationException::withMessages(['message' => trans('setup.install.could_not_establish_db_connection')]);
        }

        if (request('db_imported')) {
            $migrations = array();
            foreach (\File::allFiles(base_path('/database/migrations')) as $file) {
                $migrations[] = basename($file, '.php');
            }
            $db_migrations = \DB::table('migrations')->get()->pluck('migration')->all();
            if (array_diff($migrations, $db_migrations)) {
                throw ValidationException::withMessages(['message' => trans('setup.install.db_import_mismatch')]);
            }
        } else {
            $count_table_query = mysqli_query($link, "show tables");
            $count_table = mysqli_num_rows($count_table_query);
            if ($count_table) {
                throw ValidationException::withMessages(['message' => trans('setup.install.table_exist_in_database')]);
            }
        }

        if (! request('skip_db_version_check')) {
            $version_query = mysqli_query($link, 'SHOW VARIABLES where Variable_name = "version"');
            $version = $version_query->fetch_assoc();
            $this->checkDbVersion(Arr::get($version, 'Value', '1.0.0'));
        }

        return true;
    }

    /**
     * Install the script
     */
    public function install() : void
    {
        $url = config('app.verifier').'/api/cc?a=install&u='.url()->current().'&ac='.request('access_code').'&i='.config('app.item').'&e='.request('envato_email');

        $response = Http::get($url);

        if (! Arr::get($response, 'status')) {
            throw ValidationException::withMessages(['message' => Arr::get($response, 'message')]);
        }
        $checksum = Arr::get($response, 'checksum');

        $this->setDBEnv();

        $this->migrateDB();

        $this->populateRoleAndPermission();

        $this->populatePages();

        $this->makeAdmin();

        $this->setDomainConfig();

        $this->setNotificationConfig();

        $this->setSoketiApp();

        SysHelper::setApp(['INSTALLED' => $checksum]);
        SysHelper::setApp(['ACCESS_CODE' => request('access_code')]);
        SysHelper::setApp(['EMAIL' => request('envato_email')]);
        SysHelper::setEnv(['APP_ENV' => 'production']);

        if (\File::exists(public_path('storage'))) {
            \File::deleteDirectory(public_path('storage'));
        }

        \Artisan::call('storage:link');
    }

    /**
     * Write to env file
     */
    private function setDBEnv() : void
    {
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";

        $host = Arr::get(parse_url($_SERVER['HTTP_HOST']), 'path');
        $is_ip = IpHelper::isIp($host);
        $host = str_replace('www.', '', $host);
        $session_domain = $is_ip ? $host : ('.' . $host);
        $sanctum_stateful_domain = $is_ip ? $host : ($host. ',www.' . $host);

        SysHelper::setEnv([
            'APP_URL'     => $protocol.$host,
            'DB_PORT'     => request('db_port'),
            'DB_HOST'     => request('db_host'),
            'DB_DATABASE' => request('db_database'),
            'DB_USERNAME' => request('db_username'),
            'DB_PASSWORD' => request('db_password'),
            'SOKETI_DB_MYSQL_PORT'     => request('db_port'),
            'SOKETI_DB_MYSQL_HOST'     => request('db_host'),
            'SOKETI_DB_MYSQL_DATABASE' => request('db_database'),
            'SOKETI_DB_MYSQL_USERNAME' => request('db_username'),
            'SOKETI_DB_MYSQL_PASSWORD' => request('db_password'),
            'SESSION_DOMAIN' => $session_domain,
            'SANCTUM_STATEFUL_DOMAINS' => $sanctum_stateful_domain,
        ]);

        config([
            'app.env' => 'local',
            'app.url' => $protocol.$host
        ]);
        config(['telescope.enabled' => false]);

        \DB::purge('mysql');

        config([
            'database.connections.mysql.host' => request('db_host'),
            'database.connections.mysql.port' => request('db_port'),
            'database.connections.mysql.database' => request('db_database'),
            'database.connections.mysql.username' => request('db_username'),
            'database.connections.mysql.password' => request('db_password')
        ]);

        \DB::reconnect('mysql');
    }

    /**
     * Mirage tables to database
     */
    private function migrateDB() : void
    {
        if (! request('db_imported')) {
            \Artisan::call('migrate', ['--force' => true]);
        }

        \Artisan::call('key:generate', ['--force' => true]);
    }

    /**
     * Populate default roles
     */
    private function populateRoleAndPermission() : void
    {
        \Artisan::call('db:seed', ['--force' => true, '--class' => 'RoleSeeder']);
        \Artisan::call('db:seed', ['--force' => true, '--class' => 'PermissionSeeder']);
        \Artisan::call('db:seed', ['--force' => true, '--class' => 'AssignPermissionSeeder']);
    }

    /**
     * Populate default roles
     */
    private function populatePages() : void
    {
        \Artisan::call('db:seed', ['--force' => true, '--class' => 'SitePageTemplateSeeder']);
        \Artisan::call('db:seed', ['--force' => true, '--class' => 'SitePageSeeder']);
    }

    /**
     * Insert default admin details
     */
    private function makeAdmin() : void
    {
        $user = new User;
        $user->email = request('email');
        $user->name = request('name');
        $user->username = request('username');
        $user->uuid = Str::uuid();
        $user->password = bcrypt(request('password', 'password'));
        $user->status = 'activated';
        $user->email_verified_at = now();
        $user->save();
        $user->assignRole('admin');
    }

    /**
     * Set domain config
     */
    private function setDomainConfig() : void
    {
        Config::forceCreate([
            'name' => 'domain',
            'value' => array(
                'app' => config('app.url')
            )
        ]);
    }

    /**
     * Set notification config
     */
    private function setNotificationConfig() : void
    {
        $keys = VAPID::createVapidKeys();

        $config = Config::firstOrCreate(['name' => 'notification']);
        $value = array(
            'vapid_public_key' => Arr::get($keys, 'publicKey'),
            'vapid_private_key' => Arr::get($keys, 'privateKey')
        );
        $config->value = $value;
        $config->save();
    }

    private function setSoketiApp() : void
    {
        \Artisan::call('create:soketi-app');
    }
}
